//
//  FaceView.m
//  FaceTest
//
//  Created by Logan Collins on 4/2/12.
//  Copyright (c) 2012 Panic, Inc. All rights reserved.
//

#import "FaceView.h"
#import <QuartzCore/QuartzCore.h>


@implementation FaceView {
    CIDetector *_faceDetector;
	NSArray *_faces;
}

- (id)initWithFrame:(NSRect)frameRect {
    self = [super initWithFrame:frameRect];
    if (self) {
        _faceDetector = [CIDetector detectorOfType:CIDetectorTypeFace context:nil options:[NSDictionary dictionaryWithObjectsAndKeys:CIDetectorAccuracyHigh, CIDetectorAccuracy, nil]];
        
        [self addObserver:self forKeyPath:@"image" options:(NSKeyValueObservingOptionNew) context:nil];
    }
    return self;
}

- (id)initWithCoder:(NSCoder *)aDecoder {
    self = [super initWithCoder:aDecoder];
    if (self) {
        _faceDetector = [CIDetector detectorOfType:CIDetectorTypeFace context:nil options:[NSDictionary dictionaryWithObjectsAndKeys:CIDetectorAccuracyHigh, CIDetectorAccuracy, nil]];
        
        [self addObserver:self forKeyPath:@"image" options:(NSKeyValueObservingOptionNew) context:nil];
    }
    return self;
}

- (void)dealloc {
    [self removeObserver:self forKeyPath:@"image"];
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context {
    if ([keyPath isEqualToString:@"image"]) {
		// Create a Core Graphics image
		CGImageRef cgImage = [self.image CGImageForProposedRect:NULL context:[NSGraphicsContext currentContext] hints:nil];
		
		// Detect the faces and cache
		_faces = [_faceDetector featuresInImage:[CIImage imageWithCGImage:cgImage]];
		
		// Redisplay
        [self setNeedsDisplay:YES];
    }
    else [super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
}

- (void)drawRect:(NSRect)dirtyRect {
	CGContextRef c = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
	
	if (self.image != nil) {
		// Create an NSImage representation of the image
		CGFloat scaleFactor = 1.0;
		
		CGSize targetSize = NSSizeToCGSize([self bounds].size);
		CGSize imageSize = NSSizeToCGSize([self.image size]);
		
		CGFloat widthDelta = imageSize.width - targetSize.width;
		CGFloat heightDelta = imageSize.height - targetSize.height;
		
		if (widthDelta >= heightDelta) {
			// Scale to width
			scaleFactor = targetSize.width / imageSize.width;
		}
		else {
			// Scale to height
			scaleFactor = targetSize.height / imageSize.height;
		}
		
		CGRect scaleRect = CGRectMake(0.0, 0.0, imageSize.width * scaleFactor, imageSize.height * scaleFactor);
		[self.image drawInRect:NSRectFromCGRect(scaleRect) fromRect:NSZeroRect operation:NSCompositeSourceOver fraction:1.0];
		
		// Iterate the detected faces
		for (CIFaceFeature *face in _faces) {
			// Get the bounding rectangle of the face
			CGRect bounds = face.bounds;
			
			[[NSColor redColor] set];
			CGContextStrokeRect(c, CGRectMake(bounds.origin.x * scaleFactor, bounds.origin.y * scaleFactor, bounds.size.width * scaleFactor, bounds.size.height * scaleFactor));
			
			// Get the position of facial features
			if (face.hasLeftEyePosition) {
				CGPoint leftEyePosition = face.leftEyePosition;
				
				[[NSColor blueColor] set];
				CGContextStrokeRect(c, CGRectMake(leftEyePosition.x * scaleFactor - 10.0, leftEyePosition.y * scaleFactor - 10.0, 20.0, 20.0));
			}
			
			if (face.hasRightEyePosition) {
				CGPoint rightEyePosition = face.rightEyePosition;
				
				[[NSColor blueColor] set];
				CGContextStrokeRect(c, CGRectMake(rightEyePosition.x * scaleFactor - 10.0, rightEyePosition.y * scaleFactor - 10.0, 20.0, 20.0));
			}
			
			if (face.hasMouthPosition) {
				CGPoint mouthPosition = face.mouthPosition;
				
				[[NSColor blueColor] set];
				CGContextStrokeRect(c, CGRectMake(mouthPosition.x * scaleFactor - 10.0, mouthPosition.y * scaleFactor - 10.0, 20.0, 20.0));
			}
		}
	}
}

@end
